/*******************************************************************************
 * Copyright (c) 2008, 2012 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.acceleo.parser.tests.cst;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.fail;

import java.io.File;

import org.eclipse.acceleo.internal.parser.cst.CSTParser;
import org.eclipse.acceleo.parser.AcceleoSourceBuffer;

/**
 * This class will provide all CSTParserTest classes with utility method to check warnings, problems and infos
 * generated by the parsing.
 * 
 * @author <a href="mailto:stephane.begaudeau@obeo.fr">Stephane Begaudeau</a>
 */
public abstract class AbstractCSTParserTests {

	/**
	 * Check the parsing problems, warnings and infos.
	 * 
	 * @param buffer
	 *            The string buffer
	 * @param infoCount
	 *            The info count
	 * @param warningCount
	 *            The warning count
	 * @param problemCount
	 *            The problem count
	 */
	protected void checkCSTParsing(StringBuffer buffer, int infoCount, int warningCount, int problemCount) {
		AcceleoSourceBuffer source = new AcceleoSourceBuffer(buffer);
		checkCSTParsing(source, infoCount, warningCount, problemCount);
	}

	/**
	 * Check the parsing problems, warnings and infos.
	 * 
	 * @param file
	 *            The file
	 * @param infoCount
	 *            The info count
	 * @param warningCount
	 *            The warning count
	 * @param problemCount
	 *            The problem count
	 */
	protected void checkCSTParsing(File file, int infoCount, int warningCount, int problemCount) {
		AcceleoSourceBuffer source = new AcceleoSourceBuffer(file);
		checkCSTParsing(source, infoCount, warningCount, problemCount);
	}

	/**
	 * Check the parsing problems, warnings and infos.
	 * 
	 * @param source
	 *            The source buffer
	 * @param infoCount
	 *            The info count
	 * @param warningCount
	 *            The warning count
	 * @param problemCount
	 *            The problem count
	 */
	private void checkCSTParsing(AcceleoSourceBuffer source, int infoCount, int warningCount, int problemCount) {
		CSTParser parser = new CSTParser(source);
		assertNotNull(parser.parse());
		checkProblems(source, problemCount);
		checkWarnings(source, warningCount);
		checkInfos(source, infoCount);
	}

	protected void checkCSTInvalidParsing(StringBuffer buffer, int infoCount, int warningCount, int problemCount) {
		AcceleoSourceBuffer source = new AcceleoSourceBuffer(buffer);
		checkCSTInvalidParsing(source, infoCount, warningCount, problemCount);
	}

	protected void checkCSTInvalidParsing(File file, int infoCount, int warningCount, int problemCount) {
		AcceleoSourceBuffer source = new AcceleoSourceBuffer(file);
		checkCSTInvalidParsing(source, infoCount, warningCount, problemCount);
	}

	private void checkCSTInvalidParsing(AcceleoSourceBuffer source, int infoCount, int warningCount,
			int problemCount) {
		CSTParser parser = new CSTParser(source);
		assertNull(parser.parse());
		checkProblems(source, problemCount);
		checkWarnings(source, warningCount);
		checkInfos(source, infoCount);
	}

	/**
	 * Checks the number of problems in the source buffer.
	 * 
	 * @param source
	 *            The source buffer
	 * @param problemsCount
	 *            The number of problems wanted
	 */
	protected void checkProblems(AcceleoSourceBuffer source, int problemsCount) {
		if (source.getProblems().getList().size() != problemsCount) {
			fail("You must have " + problemsCount + " syntax errors : " + source.getProblems().getList().size()); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	/**
	 * Checks the number of warnings in the source buffer.
	 * 
	 * @param source
	 *            The source buffer
	 * @param problemsCount
	 *            The number of warnings wanted
	 */
	protected void checkWarnings(AcceleoSourceBuffer source, int warningsCount) {
		if (source.getWarnings().getList().size() != warningsCount) {
			fail("You must have " + warningsCount + " syntax warnings : " + source.getWarnings().getList().size()); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	/**
	 * Checks the number of infos in the source buffer.
	 * 
	 * @param source
	 *            The source buffer
	 * @param problemsCount
	 *            The number of infos wanted
	 */
	protected void checkInfos(AcceleoSourceBuffer source, int infosCount) {
		if (source.getInfos().getList().size() != infosCount) {
			fail("You must have " + infosCount + " syntax infos : " + source.getInfos().getList().size()); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
}
