/*******************************************************************************
 * Copyright (c) 2009, 2012 Obeo.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Obeo - initial API and implementation
 *******************************************************************************/
package org.eclipse.acceleo.engine.tests.unit.extensibility.dynamicoverride;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.acceleo.common.utils.ModelUtils;
import org.eclipse.acceleo.engine.generation.strategy.DefaultStrategy;
import org.eclipse.acceleo.engine.generation.strategy.PreviewStrategy;
import org.eclipse.acceleo.engine.service.AcceleoService;
import org.eclipse.acceleo.engine.tests.AcceleoEngineTestPlugin;
import org.eclipse.acceleo.engine.tests.unit.AbstractAcceleoTest;
import org.eclipse.acceleo.model.mtl.Module;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.FrameworkEvent;
import org.osgi.framework.FrameworkListener;
import org.osgi.framework.ServiceReference;
import org.osgi.service.packageadmin.PackageAdmin;

/**
 * This will be used to test the invocation of Java services from Acceleo.
 * 
 * @author <a href="mailto:laurent.goubet@obeo.fr">Laurent Goubet</a>
 */
@SuppressWarnings("nls")
@Ignore
public class AcceleoDynamicOverridesTest extends AbstractAcceleoTest {
	/** Id of the bundle containing dynamic overrides for this test. */
	private static final String DYNAMIC_OVERRIDES_PLUGIN_ID = "org.eclipse.acceleo.dynamic.tests";

	/** The bundle context will allow us to (un)install bundles. */
	private BundleContext context;

	/** This will allow us to (un)install the dynamic overrides bundle when needed. */
	private Bundle dynamicOverridesBundle;

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.acceleo.engine.tests.unit.AbstractAcceleoTest#getModuleLocation()
	 */
	@Override
	public String getModuleLocation() {
		// Useless as we'll override setUp
		return "";
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.acceleo.engine.tests.unit.AbstractAcceleoTest#getReferencePath()
	 */
	@Override
	public String getReferencePath() {
		// We won't be generating anything from these tests
		return "";
	}

	/**
	 * Tests that calling a dynamically overriden template generates the text and files defined in the
	 * overriding template.
	 * 
	 * @throws IOException
	 *             Thrown when the output cannot be saved.
	 */
	@Test
	public void testDynamicOverrideTemplateInvocation() throws IOException {
		final Map<String, String> preview = new AcceleoService(previewStrategy).doGenerate(module,
				"entryPoint", inputModel, null, null);
		assertNotNull("A preview should have been generated by the service", preview);
		assertFalse("The preview shouldn't be empty", preview.isEmpty());

		assertSame("Unexpected number of files created.", 3, preview.size());
		final Iterator<Map.Entry<String, String>> entryIterator = preview.entrySet().iterator();
		boolean foundSuper = false;
		boolean foundOverriding = false;
		boolean foundImported = false;
		while (entryIterator.hasNext()) {
			final Map.Entry<String, String> previewEntry = entryIterator.next();
			final String writer = previewEntry.getValue();
			if (writer == null) {
				fail("A file should have been generated.");
			}
			assert writer != null;
			if ("overridingMockTemplate".equals(previewEntry.getKey())) {
				foundOverriding = true;
				assertEquals("Unexpected content for the generated file.", "overriding mock template output",
						writer.toString().trim());
			} else if ("mockTemplate".equals(previewEntry.getKey())) {
				foundSuper = true;
				assertEquals("Unexpected content for the generated file.", "mock template output", writer
						.toString().trim());
			} else if ("importedMockTemplate".equals(previewEntry.getKey())) {
				foundImported = true;
				assertEquals("Unexpected content for the generated file.", "imported mock template output",
						writer.toString().trim());
			} else {
				fail("unexpected name of generated file : " + previewEntry.getKey());
			}
		}
		assertTrue("Super hasn't been called from dynamic override.", foundSuper);
		assertTrue("Dynamic override hasn't been called.", foundOverriding);
		assertTrue("Dynamically imported hasn't been called.", foundImported);
	}

	/**
	 * Tests that calling a dynamically overriden template generates its own text and not the overriding one
	 * when the overriding is located in a removed plugin.
	 * 
	 * @throws IOException
	 *             Thrown when the output cannot be saved.
	 */
	@Test
	public void testDynamicOverrideRemovedPlugin() {
		try {
			dynamicOverridesBundle.uninstall();
			refreshPackages(null);
		} catch (BundleException e) {
			fail("Couldn't stop dynamic overrides plugin");
		}

		final Map<String, String> preview = new AcceleoService(previewStrategy).doGenerate(module,
				"entryPoint", inputModel, null, null);
		assertNotNull("A preview should have been generated by the service", preview);
		assertFalse("The preview shouldn't be empty", preview.isEmpty());

		assertSame("Unexpected number of files created.", 1, preview.size());
		final Map.Entry<String, String> previewEntry = preview.entrySet().iterator().next();
		assertEquals("unexpected name of generated file", "mockTemplate", previewEntry.getKey());
		final String writer = previewEntry.getValue();
		if (writer != null) {
			assertEquals("Unexpected content for the generated file.", "mock template output", writer
					.toString().trim());
		} else {
			fail("A file should have been generated.");
		}
	}

	/**
	 * Tests that calling a dynamically overriden template generates the overriding's text when the overriding
	 * template is located in a removed then reinstalled plugin.
	 * 
	 * @throws IOException
	 *             Thrown when the output cannot be saved.
	 */
	@Test
	public void testDynamicOverrideReinstalledPlugin() {
		try {
			dynamicOverridesBundle.uninstall();
			refreshPackages(null);
			if (dynamicOverridesBundle.getState() == Bundle.UNINSTALLED) {
				dynamicOverridesBundle = context.installBundle(dynamicOverridesBundle.getLocation());
				refreshPackages(new Bundle[] {dynamicOverridesBundle, });
			}
		} catch (BundleException e) {
			fail("Couldn't stop dynamic overrides plugin");
		}
		while (true) {
			if (dynamicOverridesBundle.getState() == Bundle.RESOLVED) {
				break;
			}
		}

		/*
		 * We could use a registry listener to make sure the contribution gets refreshed ... For the purpose
		 * of unit tests we'll simply wait for some seconds.
		 */
		final Object lock = new Object();
		synchronized(lock) {
			try {
				lock.wait(1000L);
			} catch (InterruptedException e) {
				// Shouldn't happen. Either way, this was just a "waiting" block
			}
		}

		final Map<String, String> preview = new AcceleoService(previewStrategy).doGenerate(module,
				"entryPoint", inputModel, null, null);
		assertNotNull("A preview should have been generated by the service", preview);
		assertFalse("The preview shouldn't be empty", preview.isEmpty());

		assertSame("Unexpected number of files created.", 3, preview.size());
		final Iterator<Map.Entry<String, String>> entryIterator = preview.entrySet().iterator();
		boolean foundSuper = false;
		boolean foundOverriding = false;
		boolean foundImported = false;
		while (entryIterator.hasNext()) {
			final Map.Entry<String, String> previewEntry = entryIterator.next();
			final String writer = previewEntry.getValue();
			if (writer == null) {
				fail("A file should have been generated.");
			}
			assert writer != null;
			if ("overridingMockTemplate".equals(previewEntry.getKey())) {
				foundOverriding = true;
				assertEquals("Unexpected content for the generated file.", "overriding mock template output",
						writer.toString().trim());
			} else if ("mockTemplate".equals(previewEntry.getKey())) {
				foundSuper = true;
				assertEquals("Unexpected content for the generated file.", "mock template output", writer
						.toString().trim());
			} else if ("importedMockTemplate".equals(previewEntry.getKey())) {
				foundImported = true;
				assertEquals("Unexpected content for the generated file.", "imported mock template output",
						writer.toString().trim());
			} else {
				fail("unexpected name of generated file : " + previewEntry.getKey());
			}
		}
		assertTrue("Super hasn't been called from dynamic override.", foundSuper);
		assertTrue("Dynamic override hasn't been called.", foundOverriding);
		assertTrue("Dynamically imported hasn't been called.", foundImported);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.acceleo.engine.tests.unit.AbstractAcceleoTest#tearDown()
	 */
	@After
	@Override
	public void tearDown() {
		super.tearDown();
		if (dynamicOverridesBundle.getState() == Bundle.UNINSTALLED) {
			try {
				dynamicOverridesBundle = context.installBundle(dynamicOverridesBundle.getLocation());
				refreshPackages(new Bundle[] {dynamicOverridesBundle, });
			} catch (BundleException e) {
				fail(e.getMessage());
			}
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see org.eclipse.acceleo.engine.tests.unit.AbstractAcceleoTest#setUp()
	 */
	@Before
	@Override
	public void setUp() {
		// Shortcuts inherited behavior
		final String mockModulePath = "data/dynamicoverride/mockModule.emtl";
		final URI mockModuleURI = URI.createPlatformPluginURI('/' + AcceleoEngineTestPlugin.PLUGIN_ID + '/'
				+ mockModulePath, true);
		Resource moduleResource = ModelUtils.createResource(mockModuleURI, resourceSet);
		final Map<String, String> options = new HashMap<String, String>();
		try {
			moduleResource.load(options);
		} catch (IOException e) {
			fail(e.getMessage());
		}
		if (moduleResource.getContents().size() > 0) {
			module = (Module)moduleResource.getContents().get(0);
		} else {
			fail("Couldn't load mockModule");
		}
		dynamicOverridesBundle = Platform.getBundle(DYNAMIC_OVERRIDES_PLUGIN_ID);
		context = Platform.getBundle(org.eclipse.acceleo.engine.tests.AcceleoEngineTestPlugin.PLUGIN_ID)
				.getBundleContext();

		defaultStrategy = new DefaultStrategy();
		previewStrategy = new PreviewStrategy();
	}

	/**
	 * Refreshes all exported packages of the given bundles. This must be called after installing the bundle.
	 * 
	 * @param bundles
	 *            Bundles which exported packages are to be refreshed.
	 */
	void refreshPackages(Bundle[] bundles) {
		ServiceReference packageAdminReference = context.getServiceReference(PackageAdmin.class.getName());
		PackageAdmin packageAdmin = null;
		if (packageAdminReference != null) {
			packageAdmin = (PackageAdmin)context.getService(packageAdminReference);
		}

		if (packageAdmin != null) {
			final boolean[] flag = new boolean[] {false, };
			FrameworkListener listener = new FrameworkListener() {
				public void frameworkEvent(FrameworkEvent event) {
					if (event.getType() == FrameworkEvent.PACKAGES_REFRESHED) {
						synchronized(flag) {
							flag[0] = true;
							flag.notifyAll();
						}
					}
				}
			};

			context.addFrameworkListener(listener);
			packageAdmin.refreshPackages(bundles);
			synchronized(flag) {
				while (!flag[0]) {
					try {
						flag.wait();
					} catch (InterruptedException e) {
						// discard
						break;
					}
				}
			}
			context.removeFrameworkListener(listener);
			context.ungetService(packageAdminReference);
		}
	}
}
